const BLACK = CreateColor(0,0,0);
const RED = CreateColor(255,0,0);
const ORANGE = CreateColor(194, 147, 80);
const DARKORANGE = CreateColor(48, 36, 20);

const NEXTBOX = LoadImage("NextBox.png");

/*
 * xLength: the x length of the board you want.
 * yLength: the y length of the board you want.
 * interval: the interval (in milliseconds) the block falls down.
 * maxHolesNeeded: the number of holes you must have at maximum to complete the level.
 * yLocOfLine: the y location of the red line where you count the holes at.
 * representationOfBoard: the String representation of the board if you want a custom board (with gray blockers)
 */

function BrickBoard(interval, enemyHPViscosity, enemyFace, enemyFaceHurt, enemyFaceHappy, representationOfBoard, endless) {
	// initialize board
	this.board = representationOfBoard;
	this.endless = endless || false;
	
	// get the next few pieces.
	this.currentPiece = this.randomPiece();
	this.currentPiece.x = Math.floor(this.board.length / 2 - this.currentPiece.xLength() / 2);
	this.nextPiece = [ this.randomPiece(), this.randomPiece(), this.randomPiece(), this.randomPiece(), this.randomPiece()];
	this.interval = (interval < 100) ? 100 : interval;
	this.time = 0;
	
	this.xLen = this.board.length;
	this.yLen = this.board[0].length;
	
	this.xOffset = SCREENWIDTH / 2 - this.xLen * 5;
	this.yOffset = SCREENHEIGHT / 2 - this.yLen * 5;
	
	this.enemyHP = 40;
	this.enemyHPHeight = 40;
	this.enemyHPUpdateTime = 0;
	this.enemyHPViscosity = (enemyHPViscosity > 0) ? enemyHPViscosity : 1;
	this.enemyAttackTime = 0;
	
	this.enemyFace = enemyFace;
	this.enemyFaceHurt = enemyFaceHurt;
	this.enemyFaceHappy = enemyFaceHappy;
	this.enemyCurrentFace = enemyFace;
	this.enemyFaceTime = 0;
	
	this.yourFace = EARLNEUTRAL;
	
	this.screenXOffset = 0;
	this.screenYOffset = 0;
	this.shakeTime = 0;
	this.lastMoved = 0;
	
	this.wonGame = true;
	this.forceExit = false; // quit game from Pause screen
}

BrickBoard.prototype.createAttackTime = function() {
	return GetTime() + Math.pow(Math.log(this.interval) / Math.log(10), 2) * 5000 + Math.floor(Math.random() * 5000);
}

BrickBoard.prototype.play = function() {
	this.time = GetTime() + this.interval;
	CreateBoardKeys();
	var tmpTime = GetTime();
	while(tmpTime + 1000 > GetTime()) {
		this.blit();
		THREECOUNT.blit(0, 0);
		FlipScreen();
	}
	var tmpTime = GetTime();
	while(tmpTime + 1000 > GetTime()) {
		this.blit();
		TWOCOUNT.blit(0, 0);
		FlipScreen();
	}
	var tmpTime = GetTime();
	while(tmpTime + 1000 > GetTime()) {
		this.blit();
		ONECOUNT.blit(0, 0);
		FlipScreen();
	}
	this.enemyAttackTime = this.createAttackTime() + 5000;
	while(!this.isDone()) {
		this.handlePieces();
		
		if(!IsKeyPressed(Keys.LEFT))
			this.currentPiece.resetLeftTimes();
		if(!IsKeyPressed(Keys.RIGHT))
			this.currentPiece.resetRightTimes();
		
		for(var key in Keys) {
			if(IsKeyPressed(Keys[key]))
				KeyHandler.addKey(Keys[key]);
		}
		
		KeyHandler.run();
		this.blit();
		FlipScreen();
	}
	KeyHandler.removeAll();
	
	if(!this.wonGame) {
		this.blit();
		ApplyColorMask(CreateColor(0, 0, 0, 128));
		FlipScreen();
		Delay(2000);
		return false;
	}
	return true;
}

BrickBoard.prototype.parseStringIntoArray = function(str)
{
	var length = str.substring(0,str.indexOf("\n")).length;
	var arr = new Array(length);
	
	var index = 0;
	var count = 0;
	while((index = str.indexOf("\n", index)) != -1)
	{
		index++;
		count++;
	}
	for(var i=0;i<arr.length;i++)
	{
		arr[i] = new Array(count);
	}
	
	var xLoc = 0;
	for(var y=0;y<arr[0].length;y++)
	{
		for(var x=0;x<arr.length;x++)
		{
			arr[x][y] = str.charAt(xLoc);
			xLoc++;
		}
		xLoc++;
	}
	return arr;
}

BrickBoard.prototype.pause = function() {
	this.blit();
	ApplyColorMask(CreateColor(0, 0, 0, 128));
	var img = GrabImage(0, 0, 320, 240);
	
	var x = GetScreenWidth() * 0.5 - 30;
	var y = GetScreenHeight() / 2;
	
	var time = GetTime();
	var sel = 0;
	var pressed = false;
	while(true) {
		if(IsKeyPressed(Keys.ACCEPT)) {
			if(sel == 0) {
				break;
			} else {
				this.wonGame = false;
				this.forceExit = true;
				break;
			}
		}
		if(pressed) {
			if(!IsAnyKeyPressed())
				pressed = false;
		} else if(!pressed && (IsKeyPressed(KEY_UP) || IsKeyPressed(KEY_DOWN))) {
			pressed = true;
			sel = (sel + 1) % 2;
		}
		
		img.blit(0, 0);
		WINDOW.drawWindow(x, y, 90, 24);
		FONT.setColorMask(WHITE);
		if(sel == 0)
			FONT.setColorMask(GOLD);
		DrawText(x, y, "Continue");
		FONT.setColorMask(WHITE);
		if(sel == 1)
			FONT.setColorMask(GOLD);
		DrawText(x, y + 12, "Quit");
		
		FlipScreen();
	}
	EatKey();
	
	FONT.setColorMask(WHITE);
	time = GetTime() - time;
	this.enemyAttackTime += time;
	this.time += time;
	this.enemyFaceTime += time;
}

BrickBoard.prototype.handlePieces = function() {
	if(!this.endless) {
		if(this.enemyFaceTime < GetTime()) {
			this.enemyCurrentFace = this.enemyFace;
			this.yourFace = EARLNEUTRAL;
		}
		if(this.enemyAttackTime < GetTime()) {
			this.enemyAttackTime = this.createAttackTime();
			this.enemyFaceTime = GetTime() + 1000;
			this.enemyCurrentFace = this.enemyFaceHappy;
			this.yourFace = EARLSKEPTICAL;
			// boost everything up
			this.currentPiece.y--;
			this.pushUp();
		}
	}
	if(this.time < GetTime()) {
		this.time = GetTime() + this.interval;
		this.forceCurrentPieceDown();
	}
}

BrickBoard.prototype.moveCurrentPieceLeft = function() {
	if(!this.isDone()) {
		if(this.currentPiece.moveLeft(this.board))
			this.lastMoved = GetTime();
	}
}

BrickBoard.prototype.moveCurrentPieceRight = function() {
	if(!this.isDone()) {
		if(this.currentPiece.moveRight(this.board))
			this.lastMoved = GetTime();
	}
}

BrickBoard.prototype.rotateLeft = function() {
	if(!this.isDone()) {
		if(this.currentPiece.rotateLeft(this.board))
			this.lastMoved = GetTime();
	}
}

BrickBoard.prototype.rotateRight = function() {
	if(!this.isDone()) {
		if(this.currentPiece.rotateRight(this.board))
			this.lastMoved = GetTime();
	}
}

BrickBoard.prototype.moveCurrentPieceFall = function() {
	if(!this.isDone()) {
		this.lastMoved = 0;
		while(!this.forceCurrentPieceDown()) {
			this.lastMoved = 0;
		}
	}
}

BrickBoard.prototype.moveCurrentPieceDown = function() {
	if(!this.isDone())
		this.currentPiece.moveDown(this.board);
}

BrickBoard.prototype.forceCurrentPieceDown = function() {
	this.currentPiece.y++;
	if(!this.currentPiece.isObstructing(this.board)) {
		this.lastMoved = GetTime();
		return false;
	}
	
	// the piece is blocked.
	this.currentPiece.y--;
	// check to see if the person moved recently.
	if(this.lastMoved + 350 > GetTime()) {
		return false;
	}
	this.assimilatePiece();
	// check to see if any full lines have been created.
	// if so, remove them.
	this.clearLines();
	
	var newPiece = this.nextPiece.shift();
	this.nextPiece[4] = this.randomPiece(0);
	
	this.currentPiece = newPiece;
	this.currentPiece.x = Math.floor(this.board.length / 2 - this.currentPiece.xLength() / 2);
	if(this.currentPiece.isObstructing(this.board))
		this.wonGame = false;
	
	this.time = GetTime() + this.interval;
	
	return true;
}

BrickBoard.prototype.clearLines = function() {
	var numEmpty = 0;
	var clearedList = [];
	for(var y = 0; y < this.yLen; y++) {
		numEmpty = 0;
		for(var x = 0; x < this.xLen; x++) {
			if(this.board[x][y] == NOBLOCK)
				numEmpty++;
		}
		if(numEmpty == 0) { // line is cleared. add y to list
			clearedList.push(y);
		}
	}
	if(clearedList.length > 0) {
		this.clearGFX(clearedList);
		
		
		var damage = Math.pow(2, clearedList.length - 1) * this.enemyHPViscosity;
		// add to score
		if(clearedList.length == 4) {
			Score += Math.floor(750 * 1000 / this.interval);
		} else if(clearedList.length == 3) {
			Score += Math.floor(190 * 1000 / this.interval);
		} else if(clearedList.length == 2) {
			Score += Math.floor(75 * 1000 / this.interval);
		} else if(clearedList.length == 1) {
			Score += Math.floor(30 * 1000 / this.interval);
		}
		
		if(Score > 9999999)
			Score = 9999999;
		
		if(damage >= 20)
			damage = 19;
		if(!this.endless) {
			this.enemyHP -= damage;
			if(this.enemyHP < 0)
				this.enemyHP = 0;
		}
		
		// speed up
		this.interval -= 5 * clearedList.length;
		if(this.interval < 100) {
			this.interval = 100;
		}
		
		this.shakeTime = GetTime() + 250;
		
		for(var ele in clearedList) {
			// push everything down
			this.moveLinesDownTo(clearedList[ele]);
		}
	}
}

BrickBoard.prototype.continueScreen = function() {
	this.blit();
	ApplyColorMask(CreateColor(0, 0, 0, 128));
	WINDOW.drawWindow(70, 95, 140, 25);
	DrawText(78, 102, "YOU LOSE!");
	var img = GrabImage(0, 0, 320, 240);
	
	var x = GetScreenWidth() * 0.5 - 30;
		var y = GetScreenHeight() / 2;
		
		var mm = new Menu(x, y, 90, 600);
		mm.addDefaultKeys(Keys.UP, Keys.DOWN, Keys.LEFT, Keys.RIGHT, Keys.ACCEPT, Keys.CANCEL);
		mm.setWindowStyle(WINDOW);
		mm.setArrow(ARROWMENU);
		mm.addText("Easy", function() {
			PlayPrologue(0);
			EatKey();
		}, FONT, CreateColor(255, 255, 255), CreateColor(255, 204, 0));
		mm.addText("Medium", function() {
			PlayPrologue(1);
			EatKey();
		}, FONT, CreateColor(255, 255, 255), CreateColor(255, 204, 0));
		mm.addText("Hard", function() {
			PlayPrologue(2);
			EatKey();
		}, FONT, CreateColor(255, 255, 255), CreateColor(255, 204, 0));
		mm.addText("Intense", function() {
			PlayPrologue(3);
			EatKey();
		}, FONT, CreateColor(255, 255, 255), CreateColor(255, 204, 0));
		mm.addText("Back", function() {
			mm.escape_function();
		}, FONT, CreateColor(255, 255, 255), CreateColor(255, 204, 0));
		mm.preRender = function() {
			BACKGROUND.blit(0, 0);
			LOGO.blit(95, 40);
			this.defaultPreRender();
		}
		mm.escapeable = function(){ return true; }
		
		mm.execute();
}

BrickBoard.prototype.clearGFX = function(clearedList) {
	var time = GetTime();
	var show = 1;
	var tmpTime = GetTime();
	// set enemy face to "hurt"
	if(!this.endless)
		this.enemyCurrentFace = this.enemyFaceHurt;
	// do the clear graphics
	while(time + 500 > GetTime()) {
		// shake the HP bar
		if(!this.endless && this.shakeTime < GetTime()) {
			this.shakeTime = GetTime() + 20;
			this.screenXOffset = (Math.random() < .5) ? Math.round(Math.random() * 2) : - Math.round(Math.random() * 2);
			this.screenYOffset = (Math.random() < .5) ? Math.round(Math.random() * 2) : - Math.round(Math.random() * 2);
		}
		
		this.blit();
		EARLAMUSED.blit(32, 40);
		if(GetTime() - tmpTime > 50) {
			tmpTime = GetTime();
			show = (show + 1) % 2;
		}
		if(show == 1) {
			for(var ele in clearedList) {
				for(var i = 0; i < this.xLen; i++) {
					WHITEBLOCK.blit(this.xOffset + i * 10, this.yOffset + clearedList[ele] * 10);
				}
			}
		}
		FlipScreen();
	}
	// restore shake offsets
	this.screenXOffset = 0;
	this.screenYOffset = 0;
	// set face back to normal
	if(!this.endless)
		this.enemyCurrentFace = this.enemyFace;
}

BrickBoard.prototype.moveLinesDownTo = function(yLine) {
	for(var y = yLine; y > 0; y--) {
		for(var x = 0; x < this.xLen; x++) {
			this.board[x][y] = this.board[x][y - 1];
		}
	}
}

BrickBoard.prototype.pushUp = function() {
	// push every line up
	for(var y = 0; y < this.yLen - 1; y++) {
		for(var x = 0; x < this.xLen; x++) {
			this.board[x][y] = this.board[x][y + 1];
		}
	}
	// check to see if it's game over for you.
	if(this.isGameOver()) {
		this.wonGame = false;
	}
		
	// now make the bottom line have a gray line with a hole wherever there's an empty block above.
	for(var x = 0; x < this.xLen; x++) {
		if(this.board[x][this.yLen - 2] != NOBLOCK) {
			this.board[x][this.yLen - 1] = GREYBLOCK;
		}
	}
}

BrickBoard.prototype.isGameOver = function() {
	for(var x = 0; x < this.xLen; x++) {
		if(this.board[x][0] != NOBLOCK)
			return true;
	}
	return false;
}

BrickBoard.prototype.blitNextPieces = function()
{
	for(var i = 0; i < this.nextPiece.length; i++) {
		NEXTBOX.blit(SCREENWIDTH - 40, 40 + i*40);
		this.nextPiece[i].zoomBlit(SCREENWIDTH - 35, 45 + i*40, .5);
	}
}

BrickBoard.prototype.assimilatePiece = function() {
	for(var y=0;y<this.currentPiece.rep[0].length;y++)
		for(var x=0;x<this.currentPiece.rep.length;x++)
			if(this.currentPiece.rep[x][y] != NOBLOCK && this.board[this.currentPiece.x + x][this.currentPiece.y + y] == NOBLOCK)
				this.board[this.currentPiece.x + x][this.currentPiece.y + y] = this.currentPiece.rep[x][y];
}

BrickBoard.prototype.randomPiece = function() {
	var randPiece = [ L, J, O, Z, T, S, I ];
	return randPiece[Math.floor(Math.random() * randPiece.length)].clone();
}

BrickBoard.prototype.isDone = function() {
	return this.enemyHPHeight <= 0 || !this.wonGame;
}

BrickBoard.prototype.updateEnemyHP = function() {
	if(this.enemyHP < this.enemyHPHeight && GetTime() - this.enemyHPUpdateTime > 50) {
		this.enemyHPUpdateTime = GetTime();
		this.enemyHPHeight--;
	}
}

BrickBoard.prototype.blitEnemyHP = function(xOff, yOff) {
	Rectangle(20 + xOff, 150 + yOff, 10, 40, DARKORANGE);
	Rectangle(20 + xOff, 150 + yOff + (40 - this.enemyHPHeight), 10, this.enemyHPHeight, ORANGE);
	HPIMG.blit(19 + this.screenXOffset, 190 + this.screenYOffset);
}

BrickBoard.prototype.blit = function() {
	BACKGROUND.blit(0, 0);
	
	// blit everything
	for(var y = 0; y < this.yLen; y++) {
		for(var x = 0; x < this.xLen; x++) {
			this.board[x][y].blit(this.xOffset + x * 10, this.yOffset + y * 10);
		}
	}
	
	this.currentPiece.blit(this.xOffset, this.yOffset);
	DrawText(270, 10, "Next:");
	if(!this.endless)
		this.updateEnemyHP();
	
	// blit score
	DrawText(8, 8, "Score:");
	DrawText(8, 22, this.padZeroes(Score, 7));
	
	// blit you!
	Rectangle(31, 39, 49, 42, BLACK);
	this.yourFace.blit(32, 40);
	
	if(!this.endless) {
		// blit enemy face
		Rectangle(19 + this.screenXOffset, 149 + this.screenYOffset, 61, 42, BLACK);
		this.blitEnemyHP(this.screenXOffset, this.screenYOffset);
		this.enemyCurrentFace.blit(32 + this.screenXOffset, 150 + this.screenYOffset);
	}
	
	this.blitNextPieces();
}

BrickBoard.prototype.padZeroes = function(number, numZeroes) {
	var tmp = "" + number;
	var str = "";
	for(var i = 0; i < numZeroes - tmp.length; i++) {
		str += "0";
	}
	return "" + str + number;
}